<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Filament\Resources\UserResource\RelationManagers;
use App\Models\Book;
use App\Models\Course;
use App\Models\Event;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\Indicator;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('first_name')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('last_name')
                    ->required()
                    ->maxLength(225),
                Forms\Components\TextInput::make('display_name')
                    ->maxLength(225),
                Forms\Components\TextInput::make('email')
                    ->email()
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('password')
                    ->password()
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public static function table(Table $table): Table
    {

        return $table
            ->columns([
                Tables\Columns\TextColumn::make('first_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('last_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('display_name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\ToggleColumn::make('status')
                    ->onColor('success')
                    ->offColor('danger'),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Filter::make('Products')
                    ->form([
                        Forms\Components\Select::make('course_id')
                            ->label('Course')
                            ->options(Course::pluck('name_en', 'id')),
                        Forms\Components\Select::make('book_id')
                            ->label('Book')
                            ->options(Book::pluck('name_en', 'id')),
                        Forms\Components\Select::make('event_id')
                            ->label('Event')
                            ->options(Event::pluck('name_en', 'id')),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        if ($data['course_id'] ?? null) {
                            $query->whereHas('orders', function ($query) use ($data) {
                                $query->where('productable_type', Course::class)
                                    ->where('productable_id', $data['course_id']);
                            });
                        }

                        if ($data['book_id'] ?? null) {
                            $query->whereHas('orders', function ($query) use ($data) {
                                $query->where('productable_type', Book::class)
                                    ->where('productable_id', $data['book_id']);
                            });
                        }

                        if ($data['event_id'] ?? null) {
                            $query->whereHas('orders', function ($query) use ($data) {
                                $query->where('productable_type', Event::class)
                                    ->where('productable_id', $data['event_id']);
                            });
                        }

                        return $query;
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];

                        if ($data['course_id'] ?? null) {
                            $course = Course::find($data['course_id']);
                            $indicators[] = Indicator::make('Course: ' . $course?->name_en);
                        }

                        if ($data['book_id'] ?? null) {
                            $book = Book::find($data['book_id']);
                            $indicators[] = Indicator::make('Book: ' . $book?->name_en);
                        }

                        if ($data['event_id'] ?? null) {
                            $event = Event::find($data['event_id']);
                            $indicators[] = Indicator::make('Event: ' . $event?->name_en);
                        }

                        return $indicators;
                    })
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('view')
                    ->label('View')
                    ->url(fn ($record) => UserResource::getUrl('show', ['record' => $record])),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
            'show' => Pages\ShowUser::route('/{record}'), // Add this line

        ];
    }

    public static function canCreate(): bool
    {
        return false;
    }

}
