<?php

namespace App\Filament\Resources;

use AmidEsfahani\FilamentTinyEditor\TinyEditor;
use App\Enums\Category\CategoryTypes;
use App\Enums\Media\MediaCollection;
use App\Filament\Resources\BookResource\Pages;
use App\Filament\Resources\BookResource\RelationManagers;
use App\Models\Book;
use Filament\Forms;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class BookResource extends Resource
{
    protected static ?string $model = Book::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';
    protected static ?string $navigationGroup = 'Products';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()->schema([
                    Forms\Components\Select::make('category_id')
                        ->relationship('category', 'name_en', function ($query){
                            $query->where('type', CategoryTypes::BOOK->value);
                        })
                        ->label('Category')
                        ->required(),
                ]),

                Forms\Components\Section::make()->columns(2)->schema([

                    Forms\Components\TextInput::make('name_ar')
                        ->label('Name (Arabic)')
                        ->required()
                        ->maxLength(255)
                        ->afterStateUpdated(
                            fn(string $operation, $state, Forms\Set $set) => $set(
                                'slug',
                                slugArabic(title: $state)
                            )
                        ),
                    Forms\Components\TextInput::make('name_en')
                        ->label('Name (English)')
                        ->required()
                        ->maxLength(255),

                    Forms\Components\TextInput::make('slug')
                        ->label('Slug')
                        ->disabled()
                        ->dehydrated()
                        ->required()
                        ->unique(Book::class, 'slug', ignoreRecord: true),

                    Forms\Components\TextInput::make('price')
                        ->label('Price')
                        ->required()
                        ->columnSpanFull()
                        ->numeric()
                        ->prefix('$'),

                    Forms\Components\TextInput::make('sale_price')
                        ->label('Sale Price')
                        ->columnSpanFull()
                        ->numeric()
                        ->prefix('$'),

                    TinyEditor::make('description_ar')
                        ->label('Description (Arabic)')
                        ->profile('default|simple|full|minimal|none|custom')
                        ->direction('auto') // Set RTL or use ->direction('auto|rtl|ltr')
                        ->columnSpan('full')
                        ->required(),

                    TinyEditor::make('description_en')
                        ->label('Description (English)')
                        ->profile('default|simple|full|minimal|none|custom')
                        ->direction('auto') // Set RTL or use ->direction('auto|rtl|ltr')
                        ->columnSpan('full')
                        ->required(),

                    SpatieMediaLibraryFileUpload::make('english_image')
                        ->label('English Book Image')
                        ->collection(MediaCollection::BOOK_IMAGE_ENGLISH->value)
                        ->columnSpanFull(),

                    SpatieMediaLibraryFileUpload::make('arabic_image')
                        ->label('Arabic Book Image')
                        ->collection(MediaCollection::BOOK_IMAGE_ARABIC->value)
                        ->columnSpanFull(),
                ]),

//                Forms\Components\Section::make('Section One')->columns(2)->schema([
//
//                    Forms\Components\TextInput::make('section_one_title_ar')
//                        ->label('Section One Title (Arabic)')
//                        ->required()
//                        ->maxLength(255),
//                    Forms\Components\TextInput::make('section_one_title_en')
//                        ->label('Section One Title (English)')
//                        ->required()
//                        ->maxLength(255),
//
//                    Forms\Components\RichEditor::make('section_one_description_ar')
//                        ->label('Section One Description (Arabic)')
//                        ->required()
//                        ->maxLength(65535)
//                        ->columnSpanFull(),
//                    Forms\Components\RichEditor::make('section_one_description_en')
//                        ->label('Section One Description (English)')
//                        ->required()
//                        ->maxLength(65535)
//                        ->columnSpanFull(),
//
//                    SpatieMediaLibraryFileUpload::make('section_one_image')
//                        ->name('section_one_image')
//                        ->label('Section One Image')
//                        ->collection(MediaCollection::BOOK_SECTION_ONE_IMAGE->value)
//                        ->columnSpanFull(),
//                ]),
//
//                Forms\Components\Section::make('Section Two')->columns(2)->schema([
//
//                    Forms\Components\TextInput::make('section_two_title_ar')
//                        ->label('Section Two Title (Arabic)')
//                        ->required()
//                        ->maxLength(255),
//                    Forms\Components\TextInput::make('section_two_title_en')
//                        ->label('Section Two Title (English)')
//                        ->required()
//                        ->maxLength(255),
//
//                    Forms\Components\RichEditor::make('section_two_description_ar')
//                        ->label('Section Two Description (Arabic)')
//                        ->required()
//                        ->columnSpanFull(),
//                    Forms\Components\RichEditor::make('section_two_description_en')
//                        ->label('Section Two Description (English)')
//                        ->required()
//                        ->columnSpanFull(),
//
//                    SpatieMediaLibraryFileUpload::make('section_two_image')
//                        ->label('Section Two Image')
//                        ->name('section_two_image')
//                        ->collection(MediaCollection::BOOK_SECTION_TWO_IMAGE->value)
//                        ->columnSpanFull(),
//                ]),
//
//                Forms\Components\Section::make('Section Three')->columns(2)->schema([
//
//                    Forms\Components\TextInput::make('section_three_title_ar')
//                        ->label('Section Three Title (Arabic)')
//                        ->required()
//                        ->maxLength(255),
//                    Forms\Components\TextInput::make('section_three_title_en')
//                        ->label('Section Three Title (English)')
//                        ->required()
//                        ->maxLength(255),
//
//                    Forms\Components\RichEditor::make('section_three_description_ar')
//                        ->label('Section Three Description (Arabic)')
//                        ->required()
//                        ->columnSpanFull(),
//                    Forms\Components\RichEditor::make('section_three_description_en')
//                        ->label('Section Three Description (English)')
//                        ->required()
//                        ->columnSpanFull(),
//
//                    SpatieMediaLibraryFileUpload::make('section_three_image')
//                        ->label('Section Three Image')
//                        ->name('section_three_image')
//                        ->collection(MediaCollection::BOOK_SECTION_THREE_IMAGE->value)
//                        ->columnSpanFull(),
//                ]),

                Forms\Components\Section::make()->columns(1)->schema([

                    SpatieMediaLibraryFileUpload::make('book')
                        ->label('Book File')
                        ->name('Book')
                        ->collection(MediaCollection::BOOK_FILE->value)
                        ->columnSpanFull(),
                ]),
                Forms\Components\Section::make()->columns(1)->schema([

                    Forms\Components\Toggle::make('status')
                        ->label('Status')
                        ->required(),
                    Forms\Components\Toggle::make('is_featured')
                        ->label('Is Featured')
                        ->required(),
                ]),
            ]);



    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name_ar')
                    ->label('Arabic Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('name_en')
                    ->label('English Name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('category.name_en')
                    ->label('English Category')
                    ->searchable(),

                Tables\Columns\ToggleColumn::make('status')
                    ->onColor('success')
                    ->offColor('danger')
                    ->searchable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([

            ]);
    }

    public static function getRelations(): array
    {
        return [
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListBooks::route('/'),
            'create' => Pages\CreateBook::route('/create'),
            'edit' => Pages\EditBook::route('/{record}/edit'),
        ];
    }

    public static function canDelete(Model $record): bool
    {
        return false;
    }
}
